<?php
session_start();

// --- LÓGICA DE INTEGRACIÓN CON EL PANEL ---
// 1. Obtener eventos de la sesión (Simulación de BD)
$todos_eventos = isset($_SESSION['eventos']) ? $_SESSION['eventos'] : [];

// 2. Filtrar solo los eventos ACTIVOS
$eventos_visibles = array_filter($todos_eventos, function($evento) {
    return isset($evento['estado']) && $evento['estado'] === 'activo';
});

// 3. Ordenar por fecha (el más próximo primero)
usort($eventos_visibles, function($a, $b) {
    return strtotime($a['fecha']) - strtotime($b['fecha']);
});

// Helper para fechas en español (Meses y Días)
function fechaEs($fecha) {
    $dias = array('DOM','LUN','MAR','MIE','JUE','VIE','SAB');
    $meses = array(1=>'ENE','FEB','MAR','ABR','MAY','JUN','JUL','AGO','SEP','OCT','NOV','DIC');
    
    $obj = new DateTime($fecha);
    $mes_idx = (int)$obj->format('n');
    
    return [
        'dia_num' => $obj->format('d'),
        'dia_nom' => $dias[$obj->format('w')],
        'mes' => $meses[$mes_idx]
    ];
}
?>
<!DOCTYPE html>
<html lang="es" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hotel Atacama Suites | El Lujo del Desierto</title>
    
    <!-- Meta SEO Básico -->
    <meta name="description" content="Hotel 5 Estrellas en Copiapó, Atacama. Lujo, gastronomía y descanso en el corazón del desierto.">

    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@300;400;500;600&family=Playfair+Display:ital,wght@0,400;0,600;0,700;1,400&display=swap" rel="stylesheet">

    <!-- Phosphor Icons -->
    <script src="https://unpkg.com/@phosphor-icons/web"></script>

    <!-- SweetAlert2 (Alertas Lindas) -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <!-- FINTOC JS SDK -->
    <script src="https://js.fintoc.com/v1/"></script>

    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        brand: {
                            dark: '#1a1a1a',      /* Gris Carbón */
                            copper: '#c17c5b',    /* Cobre Atacama */
                            lightCopper: '#dca68b', /* Cobre Claro */
                            cream: '#fdfbf7',     /* Blanco Hueso */
                            gray: '#8c8c8c'
                        }
                    },
                    fontFamily: {
                        sans: ['Montserrat', 'sans-serif'],
                        serif: ['Playfair Display', 'serif'],
                    },
                    animation: {
                        'fade-in-up': 'fadeInUp 1s ease-out forwards',
                        'fade-in': 'fadeIn 1.5s ease-out forwards',
                        'bounce-dots': 'bounceDots 1.4s infinite ease-in-out both',
                        'wiggle': 'wiggle 1s ease-in-out infinite',
                        'scale-in': 'scaleIn 0.5s ease-out forwards',
                        'float': 'float 6s ease-in-out infinite',
                    },
                    keyframes: {
                        fadeInUp: {
                            '0%': { opacity: '0', transform: 'translateY(20px)' },
                            '100%': { opacity: '1', transform: 'translateY(0)' },
                        },
                        fadeIn: {
                            '0%': { opacity: '0' },
                            '100%': { opacity: '1' },
                        },
                        scaleIn: {
                            '0%': { opacity: '0', transform: 'scale(0.95)' },
                            '100%': { opacity: '1', transform: 'scale(1)' },
                        },
                        bounceDots: {
                            '0%, 80%, 100%': { transform: 'scale(0)' },
                            '40%': { transform: 'scale(1)' },
                        },
                        float: {
                            '0%, 100%': { transform: 'translateY(0)' },
                            '50%': { transform: 'translateY(-10px)' },
                        }
                    }
                }
            }
        }
    </script>

    <style>
        /* --- PRELOADER PERSONALIZADO --- */
        #preloader {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: #1a1a1a; /* Brand Dark */
            z-index: 9999;
            display: flex;
            justify-content: center;
            align-items: center;
            transition: opacity 0.8s ease-in-out, visibility 0.8s ease-in-out;
        }
        
        #preloader.loaded {
            opacity: 0;
            visibility: hidden;
        }

        .loader-container {
            position: relative;
            width: 100px;
            height: 100px;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .loader-ring {
            position: absolute;
            width: 100%;
            height: 100%;
            border-radius: 50%;
            border: 2px solid rgba(193, 124, 91, 0.1); /* Copper suave */
            border-top-color: #c17c5b; /* Brand Copper */
            animation: spin 1.5s cubic-bezier(0.68, -0.55, 0.27, 1.55) infinite;
        }
        
        .loader-ring:nth-child(2) {
            width: 80%;
            height: 80%;
            border-top-color: transparent;
            border-bottom-color: #dca68b; /* Light Copper */
            animation-duration: 2s;
            animation-direction: reverse;
        }

        .loader-text {
            font-family: 'Playfair Display', serif;
            font-size: 1.5rem;
            color: #fdfbf7;
            font-weight: 700;
            letter-spacing: 0.1em;
            animation: pulse 2s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }

        /* --- ESTILOS DE DISEÑO MEJORADO --- */
        
        .glass-nav {
            background: rgba(26, 26, 26, 0.85);
            backdrop-filter: blur(12px);
            border-bottom: 1px solid rgba(255,255,255,0.05);
        }
        
        /* Glassmorphism para Tarjetas */
        .glass-card {
            background: rgba(255, 255, 255, 0.03);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            box-shadow: 0 4px 30px rgba(0, 0, 0, 0.1);
            transition: all 0.4s ease;
        }
        .glass-card:hover {
            background: rgba(255, 255, 255, 0.08);
            border: 1px solid rgba(193, 124, 91, 0.5); /* Brand Copper */
            transform: translateY(-5px);
            box-shadow: 0 10px 40px rgba(193, 124, 91, 0.15);
        }

        /* Botón Gradiente Cobre de Lujo */
        .btn-luxury {
            background: linear-gradient(135deg, #c17c5b 0%, #a05a3b 100%);
            position: relative;
            overflow: hidden;
            z-index: 1;
            transition: all 0.3s ease;
        }
        .btn-luxury::before {
            content: '';
            position: absolute;
            top: 0; left: 0; right: 0; bottom: 0;
            background: linear-gradient(135deg, #dca68b 0%, #c17c5b 100%);
            opacity: 0;
            z-index: -1;
            transition: opacity 0.3s ease;
        }
        .btn-luxury:hover::before {
            opacity: 1;
        }
        .btn-luxury:hover {
            box-shadow: 0 5px 15px rgba(193, 124, 91, 0.4);
            transform: translateY(-2px);
        }

        .hide-scrollbar::-webkit-scrollbar { display: none; }
        .hide-scrollbar { -ms-overflow-style: none; scrollbar-width: none; }
        
        .chat-message { opacity: 0; animation: fadeIn 0.3s forwards; }
        
        /* Slider Styles */
        .slider-image, .menu-page, .salon-slide {
            position: absolute;
            top: 0; left: 0; width: 100%; height: 100%;
            object-fit: cover;
            opacity: 0;
            transition: opacity 0.8s ease-in-out;
        }
        .slider-image.active, .menu-page.active, .salon-slide.active { opacity: 1; position: relative; }
        
        .menu-page { display: none; }
        .menu-page.active { display: block; animation: fadeIn 0.6s ease-out; }

        .room-slide {
            display: none; width: 100%; height: 100%; object-fit: cover;
            border-radius: 1rem 0 0 1rem; 
        }
        .room-slide.active { display: block; animation: fadeIn 0.5s; }
        
        @media (max-width: 768px) {
            .room-slide { border-radius: 1rem 1rem 0 0; }
        }
        
        .footer-img:hover {
            transform: scale(1.05);
            filter: brightness(1.1);
            z-index: 10;
            box-shadow: 0 10px 25px rgba(0,0,0,0.3);
        }
        
        .copper-bg {
            background-image: url('https://images.unsplash.com/photo-1605634428073-677270387e07?q=80&w=2070&auto=format&fit=crop');
            background-size: cover;
            background-position: center;
        }

        .ticket-scroll::-webkit-scrollbar { width: 6px; }
        .ticket-scroll::-webkit-scrollbar-track { background: #f9fafb; }
        .ticket-scroll::-webkit-scrollbar-thumb { background: #d1d5db; border-radius: 10px; }
        .ticket-scroll::-webkit-scrollbar-thumb:hover { background: #c17c5b; }
    </style>
</head>
<body class="bg-brand-cream text-brand-dark antialiased font-sans selection:bg-brand-copper selection:text-white">

    <!-- PRELOADER DE LUJO -->
    <div id="preloader">
        <div class="loader-container">
            <div class="loader-ring"></div>
            <div class="loader-ring"></div>
            <div class="loader-text">AS</div>
        </div>
    </div>

    <!-- NAVIGATION -->
    <nav id="navbar" class="fixed w-full z-50 transition-all duration-500 py-6 text-white border-b border-white/10">
        <div class="container mx-auto px-6 flex justify-between items-center">
            <a href="#" class="flex items-center gap-3 group">
                <div class="font-serif text-3xl font-bold tracking-[0.15em] group-hover:text-brand-lightCopper transition-colors">AS</div>
                <div class="hidden md:block border-l border-white/20 pl-4 ml-1">
                    <p class="text-[10px] tracking-[0.25em] uppercase leading-tight text-white/80 group-hover:text-white transition-colors">Hotel<br>Atacama Suites</p>
                </div>
            </a>
            <div class="hidden md:flex items-center space-x-10 text-xs tracking-[0.15em] font-medium uppercase">
                <a href="#agenda-corporativa" class="hover:text-brand-lightCopper transition-colors relative group">
                    Agenda
                    <span class="absolute -bottom-2 left-0 w-0 h-[1px] bg-brand-lightCopper transition-all duration-300 group-hover:w-full"></span>
                </a>
                <a href="#suites" class="hover:text-brand-lightCopper transition-colors relative group">
                    Habitaciones
                    <span class="absolute -bottom-2 left-0 w-0 h-[1px] bg-brand-lightCopper transition-all duration-300 group-hover:w-full"></span>
                </a>
                <a href="#gastronomia" class="hover:text-brand-lightCopper transition-colors relative group">
                    Gastronomía
                    <span class="absolute -bottom-2 left-0 w-0 h-[1px] bg-brand-lightCopper transition-all duration-300 group-hover:w-full"></span>
                </a>
                <div class="flex items-center gap-2 ml-4 border border-white/20 px-4 py-2 rounded-full cursor-pointer hover:border-brand-lightCopper hover:text-brand-lightCopper transition-all bg-white/5 backdrop-blur-sm">
                    <i class="ph ph-globe text-lg"></i>
                    <span>ES</span>
                </div>
            </div>
            <button class="md:hidden text-2xl hover:text-brand-lightCopper transition-colors"><i class="ph ph-list"></i></button>
        </div>
    </nav>

    <!-- HERO SECTION VIDEO -->
    <header class="relative h-screen w-full overflow-hidden">
        <div class="absolute inset-0 w-full h-full bg-black">
             <video autoplay muted loop playsinline class="w-full h-full object-cover opacity-70 scale-105 animate-float" style="animation-duration: 20s;">
                <source src="full.mp4" type="video/mp4">
                <source src="https://videos.pexels.com/video-files/2659105/2659105-hd_1920_1080_30fps.mp4" type="video/mp4">
                <img src="https://images.unsplash.com/photo-1566073771259-6a8506099945?q=80&w=2070&auto=format&fit=crop" class="w-full h-full object-cover">
            </video>
            <!-- Gradiente mejorado para legibilidad -->
            <div class="absolute inset-0 bg-gradient-to-b from-black/70 via-transparent to-black/80 pointer-events-none"></div>
        </div>

        <div class="relative z-10 h-full flex flex-col justify-center items-center text-center px-4 pointer-events-none">
            <div class="animate-fade-in-up" style="animation-delay: 0.1s;">
                <span class="inline-block h-[1px] w-12 bg-brand-copper mb-4 align-middle"></span>
                <span class="text-brand-lightCopper tracking-[0.4em] uppercase text-xs mx-4 align-middle">Bienvenido a Copiapó</span>
                <span class="inline-block h-[1px] w-12 bg-brand-copper mb-4 align-middle"></span>
            </div>
            
            <h1 class="font-serif text-5xl md:text-7xl lg:text-8xl text-white font-medium mb-8 animate-fade-in-up leading-tight drop-shadow-2xl" style="animation-delay: 0.3s;">
                El Lujo del<br><span class="italic font-light text-brand-cream/90">Desierto</span>
            </h1>
            
            <p class="text-gray-200 max-w-lg mx-auto font-light text-lg mb-12 animate-fade-in-up leading-relaxed" style="animation-delay: 0.5s;">
                Un oasis de sofisticación donde el diseño moderno se encuentra con la inmensidad de Atacama.
            </p>
            
            <div class="pointer-events-auto mt-4 animate-fade-in-up" style="animation-delay: 0.7s;">
                <button onclick="toggleChatbot()" class="bg-white/10 backdrop-blur-md border border-white/20 text-white px-8 py-4 rounded-full flex items-center gap-3 hover:bg-white/20 transition-all group">
                    <i class="ph ph-chat-circle-text text-xl group-hover:scale-110 transition-transform"></i>
                    <span class="uppercase text-xs tracking-widest font-bold">Consultar Dudas</span>
                </button>
            </div>
        </div>
        
        <div class="absolute bottom-10 left-1/2 transform -translate-x-1/2 text-white animate-bounce hidden md:block opacity-60">
            <i class="ph ph-caret-down text-3xl"></i>
        </div>
    </header>

    <!-- STICKY BOOKING BAR -->
    <div class="relative z-30 -mt-24 px-4 container mx-auto mb-20">
        <div class="bg-brand-dark/95 backdrop-blur-xl shadow-2xl rounded-2xl p-6 md:p-10 border-t-4 border-brand-copper transform hover:-translate-y-1 transition-transform duration-500">
            <form class="flex flex-col lg:flex-row gap-6 items-end justify-between" onsubmit="event.preventDefault(); showReservationAlert();">
                <div class="w-full lg:w-1/4">
                    <label class="block text-brand-gray text-[10px] uppercase tracking-widest mb-3 font-bold">Llegada</label>
                    <div class="flex items-center border-b border-white/10 pb-3 hover:border-brand-copper transition-colors group">
                        <i class="ph ph-calendar-blank text-brand-copper text-2xl mr-3 group-hover:text-white transition-colors"></i>
                        <input type="date" class="bg-transparent text-white w-full focus:outline-none font-serif text-xl placeholder-white/30 cursor-pointer">
                    </div>
                </div>
                <div class="w-full lg:w-1/4">
                    <label class="block text-brand-gray text-[10px] uppercase tracking-widest mb-3 font-bold">Salida</label>
                    <div class="flex items-center border-b border-white/10 pb-3 hover:border-brand-copper transition-colors group">
                        <i class="ph ph-calendar-check text-brand-copper text-2xl mr-3 group-hover:text-white transition-colors"></i>
                        <input type="date" class="bg-transparent text-white w-full focus:outline-none font-serif text-xl cursor-pointer">
                    </div>
                </div>
                <div class="w-full lg:w-1/4">
                    <label class="block text-brand-gray text-[10px] uppercase tracking-widest mb-3 font-bold">Huéspedes</label>
                    <div class="flex items-center border-b border-white/10 pb-3 hover:border-brand-copper transition-colors group">
                        <i class="ph ph-users text-brand-copper text-2xl mr-3 group-hover:text-white transition-colors"></i>
                        <select class="bg-transparent text-white w-full focus:outline-none font-serif text-xl appearance-none cursor-pointer">
                            <option class="text-black">2 Adultos</option>
                            <option class="text-black">1 Adulto</option>
                            <option class="text-black">Family (2+2)</option>
                        </select>
                        <i class="ph ph-caret-down text-white/50 text-sm"></i>
                    </div>
                </div>
                <div class="w-full lg:w-auto">
                    <button class="btn-luxury w-full text-white px-10 py-5 font-bold tracking-[0.2em] text-xs uppercase rounded-lg flex items-center justify-center gap-3 group shadow-lg">
                        Ver Disponibilidad <i class="ph ph-arrow-right text-lg group-hover:translate-x-1 transition-transform"></i>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- SECCIÓN BIENVENIDA (RESTAURADA Y MEJORADA) -->
    <section class="py-24 container mx-auto px-6">
        <div class="flex flex-col lg:flex-row items-center gap-20">
            <div class="w-full lg:w-1/2 space-y-10">
                <div>
                    <span class="text-brand-copper font-bold tracking-[0.25em] text-xs uppercase flex items-center gap-4 mb-6">
                        <span class="w-12 h-[1px] bg-brand-copper"></span> Sobre Nosotros
                    </span>
                    <h2 class="font-serif text-5xl text-brand-dark leading-tight">
                        Su refugio ejecutivo <br> <span class="italic text-brand-copper">y familiar</span> en el norte.
                    </h2>
                </div>
                <p class="text-gray-600 leading-loose font-light text-lg">
                    Estratégicamente ubicado en Copiapó, <strong class="text-brand-dark">Hotel Atacama Suites</strong> redefine la hospitalidad combinando la eficiencia que exigen los negocios con el confort que merece su descanso. Un espacio donde cada detalle ha sido curado para su bienestar.
                </p>
                <div class="flex items-center gap-6 pt-6 border-t border-gray-100">
                    <div>
                        <p class="font-serif text-2xl italic text-brand-dark">Andrea Velandia</p>
                        <p class="text-[10px] text-gray-400 uppercase tracking-widest mt-1 font-bold">Gerente General</p>
                    </div>
                    <div class="font-serif text-5xl text-brand-copper/30 ml-auto opacity-50">A.Velandia</div>
                </div>
            </div>
            <div class="w-full lg:w-1/2 relative flex justify-center">
                <div class="absolute -top-6 -left-6 w-32 h-32 border-t-[3px] border-l-[3px] border-brand-copper hidden lg:block opacity-50"></div>
                <div class="absolute -bottom-6 -right-6 w-32 h-32 border-b-[3px] border-r-[3px] border-brand-copper hidden lg:block opacity-50"></div>
                <!-- Imagen con efecto de profundidad -->
                <div class="relative z-10 group">
                    <div class="absolute inset-0 bg-brand-copper/10 rounded-full blur-3xl opacity-0 group-hover:opacity-100 transition-opacity duration-700"></div>
                    <img src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?q=80&w=988&auto=format&fit=crop" 
                         alt="Andrea Velandia, Gerente General" 
                         class="w-[400px] h-[500px] object-cover rounded-2xl shadow-2xl hover:scale-105 transition-all duration-700 ease-in-out cursor-zoom-in relative z-10"
                         onclick="openLightbox(this.src)">
                </div>
            </div>
        </div>
    </section>

    <!-- SECCIÓN AGENDA DINÁMICA (GLASS DESIGN) -->
    <section id="agenda-corporativa" class="py-24 bg-brand-dark text-white relative overflow-hidden">
        <!-- Background decorative elements -->
        <div class="absolute top-0 right-0 w-[500px] h-[500px] bg-brand-copper rounded-full mix-blend-multiply filter blur-[120px] opacity-20 animate-pulse"></div>
        <div class="absolute bottom-0 left-0 w-[600px] h-[600px] bg-[#1a2e35] rounded-full mix-blend-screen filter blur-[100px] opacity-10"></div>

        <div class="container mx-auto px-6 relative z-10">
            <div class="text-center mb-20">
                <span class="text-brand-lightCopper font-bold tracking-[0.25em] text-xs uppercase mb-4 block">Calendario Exclusivo</span>
                <h2 class="font-serif text-5xl md:text-6xl mb-6">Agenda Corporativa & Social</h2>
                <p class="text-gray-400 max-w-2xl mx-auto font-light text-lg leading-relaxed">
                    Participe en nuestras actividades diseñadas para fomentar el networking, el aprendizaje y la celebración.
                </p>
            </div>

            <?php if(empty($eventos_visibles)): ?>
                <div class="glass-card rounded-3xl p-16 text-center max-w-2xl mx-auto">
                    <div class="w-20 h-20 bg-white/5 rounded-full flex items-center justify-center mx-auto mb-6">
                        <i class="ph ph-calendar-plus text-4xl text-brand-copper"></i>
                    </div>
                    <h3 class="text-2xl font-serif mb-3">Próximamente Nueva Agenda</h3>
                    <p class="text-gray-400 mb-8 font-light">Estamos planificando experiencias únicas para el próximo mes.</p>
                    <button onclick="openQuoteModal()" class="btn-luxury px-8 py-3 rounded-full text-white font-bold uppercase text-xs tracking-widest inline-flex items-center gap-2">
                        <span>Organizar mi Evento</span>
                    </button>
                </div>
            <?php else: ?>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                    <?php foreach($eventos_visibles as $evento): 
                        $fecha = fechaEs($evento['fecha']);
                    ?>
                    <!-- Card Glass Dinámica -->
                    <div class="glass-card rounded-2xl overflow-hidden flex flex-col group h-full">
                        <div class="relative h-64 overflow-hidden">
                            <?php if(!empty($evento['imagen'])): ?>
                                <img src="<?php echo htmlspecialchars($evento['imagen']); ?>" alt="<?php echo htmlspecialchars($evento['titulo']); ?>" class="w-full h-full object-cover transition-transform duration-1000 group-hover:scale-110">
                            <?php else: ?>
                                <div class="w-full h-full bg-gradient-to-br from-gray-800 to-black flex items-center justify-center">
                                    <i class="ph ph-image text-5xl text-white/10"></i>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Fecha Flotante -->
                            <div class="absolute top-4 left-4 bg-white/95 backdrop-blur text-brand-dark rounded-xl p-3 text-center min-w-[70px] shadow-xl z-10 group-hover:-translate-y-1 transition-transform">
                                <span class="block text-[10px] font-bold uppercase text-brand-copper tracking-widest mb-1"><?php echo $fecha['mes']; ?></span>
                                <span class="block text-3xl font-serif font-bold leading-none"><?php echo $fecha['dia_num']; ?></span>
                            </div>
                            
                            <div class="absolute bottom-0 left-0 w-full bg-gradient-to-t from-black via-black/50 to-transparent p-6 pt-20">
                                <span class="bg-brand-copper/90 backdrop-blur-sm text-white text-[10px] font-bold px-3 py-1 rounded-full uppercase tracking-wider shadow-sm mb-2 inline-block">
                                    <?php echo htmlspecialchars($evento['categoria'] ?? 'Evento'); ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="p-8 flex-1 flex flex-col">
                            <h3 class="text-2xl font-serif font-bold mb-3 text-white group-hover:text-brand-lightCopper transition-colors leading-tight">
                                <?php echo htmlspecialchars($evento['titulo']); ?>
                            </h3>
                            <p class="text-gray-400 text-sm mb-6 line-clamp-3 flex-1 font-light leading-relaxed">
                                <?php echo htmlspecialchars($evento['descripcion']); ?>
                            </p>
                            
                            <div class="flex items-center justify-between mt-auto pt-6 border-t border-white/10">
                                <div class="flex flex-col">
                                    <span class="text-[10px] text-gray-500 uppercase tracking-widest mb-1">Entrada</span>
                                    <span class="text-xl font-serif font-bold text-white">
                                        <?php echo ($evento['precio'] > 0) ? '$' . number_format($evento['precio'], 0, ',', '.') : 'Gratis'; ?>
                                    </span>
                                </div>
                                <button onclick="openTicketModal('<?php echo htmlspecialchars($evento['titulo']); ?>', <?php echo $evento['precio']; ?>)" class="bg-white text-brand-dark hover:bg-brand-copper hover:text-white px-6 py-3 rounded-full flex items-center gap-2 text-[10px] font-bold uppercase tracking-widest transition-all shadow-lg hover:shadow-brand-copper/50">
                                    <span>Ticket</span> <i class="ph ph-ticket text-lg"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- CTA al final de la grilla -->
                <div class="mt-20 text-center animate-fade-in-up">
                    <p class="mb-6 text-sm text-gray-400 uppercase tracking-[0.2em] font-medium">¿Busca un espacio exclusivo para su empresa?</p>
                    <button onclick="openQuoteModal()" class="border border-white/30 text-white px-10 py-4 rounded-full hover:border-brand-copper hover:bg-brand-copper hover:text-white transition-all font-bold uppercase text-xs tracking-[0.15em] flex items-center gap-3 mx-auto">
                        <i class="ph ph-briefcase text-lg"></i> Cotizar Evento Corporativo
                    </button>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- SECCIÓN SUITES (MEJORADA) -->
    <section id="suites" class="py-24 bg-gray-50 overflow-hidden">
        <div class="container mx-auto px-6 mb-16 flex justify-between items-end">
            <div>
                <span class="text-brand-copper font-bold tracking-[0.2em] text-xs uppercase block mb-2">Confort Absoluto</span>
                <h2 class="font-serif text-5xl text-brand-dark">Nuestras Suites</h2>
            </div>
            <div class="hidden md:flex gap-3">
                <button onclick="scrollSuites('left')" class="w-14 h-14 border border-brand-dark/10 rounded-full flex items-center justify-center hover:bg-brand-dark hover:text-white transition-all hover:scale-110"><i class="ph ph-arrow-left text-xl"></i></button>
                <button onclick="scrollSuites('right')" class="w-14 h-14 border border-brand-dark/10 rounded-full flex items-center justify-center hover:bg-brand-dark hover:text-white transition-all hover:scale-110"><i class="ph ph-arrow-right text-xl"></i></button>
            </div>
        </div>

        <div id="suites-container" class="flex gap-8 overflow-x-auto px-6 pb-12 hide-scrollbar snap-x snap-mandatory">
            <!-- Card 1 -->
            <div class="min-w-[85vw] md:min-w-[40vw] lg:min-w-[30vw] snap-center group relative cursor-pointer h-[550px] rounded-3xl overflow-hidden shadow-xl bg-white hover:shadow-2xl transition-all duration-500">
                <div class="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent z-10 pointer-events-none opacity-80 group-hover:opacity-90 transition-opacity"></div>
                <img src="https://images.unsplash.com/photo-1591088398332-8a7791972843?q=80&w=1974&auto=format&fit=crop" class="w-full h-full object-cover transition-transform duration-1000 group-hover:scale-110" alt="Junior Suite">
                <div class="absolute bottom-0 left-0 p-10 z-20 w-full text-white transform translate-y-4 group-hover:translate-y-0 transition-transform duration-500">
                    <p class="text-brand-lightCopper text-xs font-bold tracking-widest mb-3 uppercase">45m² • Vista Ciudad</p>
                    <h3 class="font-serif text-4xl mb-6">Junior Suite</h3>
                    <div class="flex gap-4 opacity-0 group-hover:opacity-100 transition-opacity duration-500 delay-100">
                        <button onclick="openRoomModal('Junior Suite', '45m²', 'La elección perfecta para el viajero de negocios.')" class="bg-white/20 backdrop-blur border border-white/50 text-white px-5 py-3 text-xs uppercase tracking-widest hover:bg-white hover:text-brand-dark transition-colors rounded-lg font-bold">Detalles</button>
                        <button onclick="openRoomModal('Junior Suite', '45m²', 'La elección perfecta para el viajero de negocios.', true)" class="btn-luxury text-white px-6 py-3 text-xs uppercase tracking-widest rounded-lg font-bold">Reservar</button>
                    </div>
                </div>
            </div>

            <!-- Card 2 -->
            <div class="min-w-[85vw] md:min-w-[40vw] lg:min-w-[30vw] snap-center group relative cursor-pointer h-[550px] rounded-3xl overflow-hidden shadow-xl bg-white hover:shadow-2xl transition-all duration-500">
                <div class="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent z-10 pointer-events-none opacity-80 group-hover:opacity-90 transition-opacity"></div>
                <img src="https://images.unsplash.com/photo-1631049307264-da0ec9d70304?q=80&w=2070&auto=format&fit=crop" class="w-full h-full object-cover transition-transform duration-1000 group-hover:scale-110" alt="Master Suite">
                <div class="absolute bottom-0 left-0 p-10 z-20 w-full text-white transform translate-y-4 group-hover:translate-y-0 transition-transform duration-500">
                    <p class="text-brand-lightCopper text-xs font-bold tracking-widest mb-3 uppercase">60m² • Jacuzzi Privado</p>
                    <h3 class="font-serif text-4xl mb-6">Master Suite</h3>
                    <div class="flex gap-4 opacity-0 group-hover:opacity-100 transition-opacity duration-500 delay-100">
                        <button onclick="openRoomModal('Master Suite', '60m²', 'Nuestra suite más exclusiva con jacuzzi privado.')" class="bg-white/20 backdrop-blur border border-white/50 text-white px-5 py-3 text-xs uppercase tracking-widest hover:bg-white hover:text-brand-dark transition-colors rounded-lg font-bold">Detalles</button>
                        <button onclick="openRoomModal('Master Suite', '60m²', 'Nuestra suite más exclusiva con jacuzzi privado.', true)" class="btn-luxury text-white px-6 py-3 text-xs uppercase tracking-widest rounded-lg font-bold">Reservar</button>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- SECCIÓN GASTRONOMÍA (RESTAURADA Y MEJORADA) -->
    <section id="gastronomia" class="bg-white relative">
        <div class="flex flex-col lg:flex-row h-auto lg:h-[700px]">
            <!-- Slider Imágenes -->
            <div class="w-full lg:w-1/2 overflow-hidden relative group" id="gastronomy-slider">
                <div class="absolute inset-0 bg-black/20 z-10 group-hover:bg-transparent transition-colors duration-700"></div>
                <img src="https://images.unsplash.com/photo-1559339352-11d035aa65de?q=80&w=1974&auto=format&fit=crop" class="slider-image active transform scale-105 group-hover:scale-100 transition-transform duration-1000" alt="Gourmet 1">
                <img src="https://images.unsplash.com/photo-1504674900247-0877df9cc836?q=80&w=2070&auto=format&fit=crop" class="slider-image" alt="Gourmet 2">
                <img src="https://images.unsplash.com/photo-1514362545857-3bc16549766b?q=80&w=1974&auto=format&fit=crop" class="slider-image" alt="Gourmet 3">
            </div>
            
            <!-- Contenido Texto -->
            <div class="w-full lg:w-1/2 flex flex-col justify-center px-12 lg:px-24 py-20 bg-brand-dark text-white relative">
                <!-- Decorative Border -->
                <div class="absolute left-0 top-1/2 -translate-y-1/2 -translate-x-1/2 w-4 h-32 bg-brand-copper hidden lg:block"></div>
                
                <span class="text-brand-copper font-bold tracking-[0.25em] text-xs uppercase mb-6 flex items-center gap-4">
                    <i class="ph ph-fork-knife text-lg"></i> Experiencia Culinaria
                </span>
                <h3 class="font-serif text-5xl mb-8 leading-tight">Sabores del <br><span class="italic text-brand-lightCopper">Valle de Copiapó</span></h3>
                <p class="text-gray-400 font-light mb-10 leading-loose text-lg">
                    Una propuesta gastronómica audaz que rescata los ingredientes endémicos del desierto de Atacama, fusionándolos con técnicas de vanguardia internacional. Disfrute de nuestra cava seleccionada en un ambiente íntimo y sofisticado.
                </p>
                <button onclick="openMenuModal()" class="border border-white/20 text-white px-10 py-4 uppercase text-xs tracking-[0.2em] font-bold hover:bg-white hover:text-brand-dark transition-all w-max rounded-sm hover:shadow-[0_0_30px_rgba(255,255,255,0.1)] flex items-center gap-3">
                    Ver Carta <i class="ph ph-book-open"></i>
                </button>
            </div>
        </div>
    </section>

    <!-- SECCIÓN EVENTOS / SALONES (RESTAURADA Y MEJORADA) -->
    <section id="eventos" class="bg-brand-cream relative">
        <div class="flex flex-col lg:flex-row-reverse h-auto lg:h-[700px]">
            <!-- Slider Imágenes -->
            <div class="w-full lg:w-1/2 overflow-hidden relative group">
                <div class="absolute inset-0 bg-brand-copper/10 z-10 mix-blend-multiply"></div>
                <div id="salon-slider-container" class="w-full h-full relative">
                    <img src="https://images.unsplash.com/photo-1505373877841-8d25f7d46678?q=80&w=2012&auto=format&fit=crop" class="salon-slide active transform group-hover:scale-105 transition-transform duration-1000" id="salon-img-1" alt="Evento Default">
                    <img src="https://images.unsplash.com/photo-1519167758481-83f550bb49b3?q=80&w=1998&auto=format&fit=crop" class="salon-slide" id="salon-img-2" alt="Evento Social">
                </div>
            </div>
            
            <!-- Contenido Texto -->
            <div class="w-full lg:w-1/2 flex flex-col justify-center px-12 lg:px-24 py-20 text-brand-dark relative">
                 <!-- Decorative Border -->
                 <div class="absolute right-0 top-1/2 -translate-y-1/2 translate-x-1/2 w-4 h-32 bg-brand-dark hidden lg:block"></div>

                <span class="text-brand-copper font-bold tracking-[0.25em] text-xs uppercase mb-6 flex items-center gap-4">
                    <i class="ph ph-users-three text-lg"></i> Eventos & Negocios
                </span>
                <h3 class="font-serif text-5xl mb-8 leading-tight">Conectividad <br><span class="italic text-brand-gray">y Prestigio</span></h3>
                <p class="text-gray-600 font-light mb-10 leading-loose text-lg">
                    Infraestructura de clase mundial para sus conferencias y reuniones. Salones modulables equipados con tecnología de punta, climatización inteligente y un servicio de catering que elevará el estándar de sus encuentros corporativos.
                </p>
                <button onclick="openQuoteModal()" class="btn-luxury text-white px-10 py-4 uppercase text-xs tracking-[0.2em] font-bold w-max rounded-lg shadow-xl flex items-center gap-3">
                    Cotizar Salón <i class="ph ph-caret-right"></i>
                </button>
            </div>
        </div>
    </section>

    <!-- FOOTER RENOVADO Y MEJORADO -->
    <footer class="bg-black text-white pt-24 pb-12 border-t border-white/10 relative overflow-hidden">
        <!-- Background elements -->
        <div class="absolute top-0 left-1/4 w-[800px] h-[800px] bg-brand-copper/10 rounded-full blur-[150px] pointer-events-none"></div>

        <div class="container mx-auto px-6 relative z-10">
            <!-- Top Section -->
            <div class="flex flex-col lg:flex-row justify-between items-start gap-12 mb-20 border-b border-white/10 pb-16">
                <!-- Brand Info -->
                <div class="lg:w-1/3 space-y-6">
                    <div>
                        <span class="font-serif text-4xl font-bold tracking-[0.1em] block mb-2">AS</span>
                        <p class="text-xs uppercase tracking-[0.3em] text-brand-copper">Hotel Atacama Suites</p>
                    </div>
                    <address class="not-italic text-gray-400 text-sm leading-relaxed space-y-2">
                        <p class="flex items-center gap-3"><i class="ph ph-map-pin text-brand-copper text-lg"></i> Colipí 794, Copiapó, Chile</p>
                        <p class="flex items-center gap-3"><i class="ph ph-phone text-brand-copper text-lg"></i> +56 52 2 123 456</p>
                        <p class="flex items-center gap-3"><i class="ph ph-envelope-simple text-brand-copper text-lg"></i> reservas@atacamasuites.cl</p>
                    </address>
                    
                    <!-- Social Icons Row -->
                    <div class="flex gap-4 pt-4">
                        <a href="#" class="w-10 h-10 rounded-full border border-white/20 flex items-center justify-center hover:bg-brand-copper hover:border-brand-copper hover:text-white transition-all text-gray-400">
                            <i class="ph ph-instagram-logo text-xl"></i>
                        </a>
                        <a href="#" class="w-10 h-10 rounded-full border border-white/20 flex items-center justify-center hover:bg-brand-copper hover:border-brand-copper hover:text-white transition-all text-gray-400">
                            <i class="ph ph-facebook-logo text-xl"></i>
                        </a>
                        <a href="#" class="w-10 h-10 rounded-full border border-white/20 flex items-center justify-center hover:bg-brand-copper hover:border-brand-copper hover:text-white transition-all text-gray-400">
                            <i class="ph ph-linkedin-logo text-xl"></i>
                        </a>
                    </div>
                </div>

                <!-- Quick Links -->
                <div class="lg:w-1/3 flex gap-12">
                    <div>
                        <h4 class="font-bold uppercase tracking-[0.2em] text-xs mb-6 text-brand-lightCopper">Explorar</h4>
                        <ul class="space-y-4 text-sm text-gray-400 font-medium">
                            <li><a href="#suites" class="hover:text-white transition-colors flex items-center gap-2 group"><span class="w-0 group-hover:w-2 h-[1px] bg-brand-copper transition-all"></span> Habitaciones</a></li>
                            <li><a href="#gastronomia" class="hover:text-white transition-colors flex items-center gap-2 group"><span class="w-0 group-hover:w-2 h-[1px] bg-brand-copper transition-all"></span> Gastronomía</a></li>
                            <li><a href="#eventos" class="hover:text-white transition-colors flex items-center gap-2 group"><span class="w-0 group-hover:w-2 h-[1px] bg-brand-copper transition-all"></span> Salones</a></li>
                            <li><a href="#agenda-corporativa" class="hover:text-white transition-colors flex items-center gap-2 group"><span class="w-0 group-hover:w-2 h-[1px] bg-brand-copper transition-all"></span> Agenda</a></li>
                        </ul>
                    </div>
                </div>

                <!-- Newsletter -->
                <div class="lg:w-1/3 w-full">
                    <h4 class="font-bold uppercase tracking-[0.2em] text-xs mb-6 text-brand-lightCopper">Newsletter</h4>
                    <p class="text-gray-500 text-sm mb-6">Reciba noticias exclusivas y ofertas especiales directamente en su bandeja de entrada.</p>
                    <form class="relative group">
                        <input type="email" placeholder="Su correo electrónico" class="w-full bg-white/5 border border-white/10 rounded-lg py-4 px-5 text-sm focus:outline-none focus:border-brand-copper focus:bg-white/10 transition-all text-white placeholder-gray-600">
                        <button class="absolute right-2 top-2 bottom-2 bg-brand-copper hover:bg-white hover:text-brand-dark text-white px-4 rounded-md transition-all">
                            <i class="ph ph-paper-plane-right text-lg"></i>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Instagram Grid -->
            <div class="mb-12">
                <p class="text-xs uppercase tracking-[0.2em] text-center mb-8 text-gray-500 font-bold">Síguenos en Instagram <span class="text-brand-copper">@atacamasuites</span></p>
                <div id="galeria" class="grid grid-cols-2 md:grid-cols-6 gap-4">
                    <img src="https://images.unsplash.com/photo-1618773928121-c32242e63f39?q=80&w=400&auto=format&fit=crop" class="h-32 w-full object-cover rounded-lg opacity-60 hover:opacity-100 transition-opacity cursor-pointer">
                    <img src="https://images.unsplash.com/photo-1544148103-0773bf10d330?q=80&w=400&auto=format&fit=crop" class="h-32 w-full object-cover rounded-lg opacity-60 hover:opacity-100 transition-opacity cursor-pointer">
                    <img src="https://images.unsplash.com/photo-1512918760532-3ed0006faf67?q=80&w=400&auto=format&fit=crop" class="h-32 w-full object-cover rounded-lg opacity-60 hover:opacity-100 transition-opacity cursor-pointer">
                    <img src="https://images.unsplash.com/photo-1520250497591-112f2f40a3f4?q=80&w=400&auto=format&fit=crop" class="h-32 w-full object-cover rounded-lg opacity-60 hover:opacity-100 transition-opacity cursor-pointer">
                    <img src="https://images.unsplash.com/photo-1560624052-449f5ddf0c31?q=80&w=400&auto=format&fit=crop" class="h-32 w-full object-cover rounded-lg opacity-60 hover:opacity-100 transition-opacity cursor-pointer">
                    <div class="h-32 w-full bg-white/5 rounded-lg flex flex-col items-center justify-center text-gray-500 hover:bg-white/10 hover:text-white transition-all cursor-pointer group">
                        <i class="ph ph-instagram-logo text-2xl mb-1 group-hover:scale-110 transition-transform"></i>
                        <span class="text-[10px] uppercase tracking-widest">Ver Más</span>
                    </div>
                </div>
            </div>
            
            <!-- Bottom -->
            <div class="flex flex-col md:flex-row justify-between items-center text-[10px] text-gray-500 uppercase tracking-widest font-bold">
                <p>&copy; 2025 Hotel Atacama Suites. Todos los derechos reservados.</p>
                <div class="flex gap-6 mt-4 md:mt-0">
                    <a href="#" class="hover:text-white transition-colors">Términos y Condiciones</a>
                    <a href="#" class="hover:text-white transition-colors">Política de Privacidad</a>
                </div>
            </div>
        </div>
    </footer>

    <!-- BOTONES FLOTANTES -->
    <a href="https://wa.me/56912345678" target="_blank" class="fixed bottom-8 right-24 bg-[#25D366] text-white w-14 h-14 rounded-full shadow-2xl hover:bg-[#128C7E] transition-all z-40 flex items-center justify-center group hover:-translate-y-1"><i class="ph ph-whatsapp-logo text-3xl"></i></a>
    <button onclick="toggleChatbot()" class="fixed bottom-8 right-8 btn-luxury text-white w-16 h-16 rounded-full shadow-2xl hover:scale-105 transition-all z-50 flex items-center justify-center group"><i class="ph ph-chat-teardrop-text text-3xl"></i></button>

    <!-- === MODALES Y POPUPS === -->

    <!-- === POPUP DE BIENVENIDA / EVENTOS === -->
    <div id="welcome-popup" class="fixed inset-0 z-[100] bg-black/70 backdrop-blur-sm hidden opacity-0 transition-opacity duration-500 flex items-center justify-center p-4">
        <div class="bg-brand-cream rounded-2xl shadow-2xl w-full max-w-4xl overflow-hidden flex flex-col md:flex-row relative transform scale-90 transition-transform duration-500" id="welcome-popup-content">
            <button onclick="closeWelcomePopup()" class="absolute top-4 right-4 z-20 bg-white/80 text-brand-dark w-10 h-10 rounded-full flex items-center justify-center hover:bg-brand-copper hover:text-white transition-all shadow-md text-xl">
                <i class="ph ph-x"></i>
            </button>
            <div class="w-full md:w-5/12 relative min-h-[250px] md:min-h-full">
                <img src="https://images.unsplash.com/photo-1561047029-3000c68339ca?q=80&w=1974&auto=format&fit=crop" class="absolute inset-0 w-full h-full object-cover">
                <div class="absolute inset-0 bg-brand-copper/30 mix-blend-multiply"></div>
                <div class="absolute bottom-6 left-6 text-white z-10">
                    <p class="font-serif text-3xl">Experiencias<br>Inolvidables</p>
                    <div class="w-12 h-1 bg-white mt-2"></div>
                </div>
            </div>
            <div class="w-full md:w-7/12 p-8 md:p-12 bg-white flex flex-col justify-center">
                <span class="text-brand-copper font-bold tracking-[0.2em] text-xs uppercase mb-2">Agenda Mensual</span>
                <h2 class="font-serif text-3xl md:text-4xl text-brand-dark mb-6">Próximos Eventos</h2>
                <div class="space-y-6 overflow-y-auto max-h-[300px] pr-2 custom-scrollbar">
                    <?php if(empty($eventos_visibles)): ?>
                        <div class="text-center py-6 text-gray-400">
                            <i class="ph ph-calendar-slash text-4xl mb-2"></i>
                            <p class="text-sm">Pronto anunciaremos nuevas actividades.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach($eventos_visibles as $evento): $fechaData = fechaEs($evento['fecha']); ?>
                        <div class="flex gap-4 items-start group cursor-pointer hover:bg-gray-50 p-2 rounded-lg transition-colors">
                            <div class="bg-brand-dark text-white w-14 h-14 rounded-lg flex flex-col items-center justify-center flex-shrink-0 group-hover:bg-brand-copper transition-colors">
                                <span class="text-xs uppercase font-bold"><?php echo $fechaData['dia_nom']; ?></span>
                                <span class="font-serif text-xl"><?php echo $fechaData['dia_num']; ?></span>
                            </div>
                            <div>
                                <h4 class="font-bold text-brand-dark group-hover:text-brand-copper transition-colors"><?php echo htmlspecialchars($evento['titulo']); ?></h4>
                                <p class="text-sm text-gray-500 line-clamp-2"><?php echo htmlspecialchars($evento['descripcion']); ?></p>
                                <?php if($evento['precio'] > 0): ?>
                                    <span class="text-xs font-bold text-brand-copper mt-1 block">$<?php echo number_format($evento['precio'], 0, ',', '.'); ?></span>
                                <?php else: ?>
                                    <span class="text-xs font-bold text-green-600 mt-1 block">Entrada Liberada</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                <div class="mt-8 flex gap-4">
                    <button onclick="closeWelcomePopup()" class="flex-1 border border-brand-gray text-brand-gray py-3 rounded-lg text-xs font-bold uppercase tracking-widest hover:border-brand-dark hover:text-brand-dark transition-colors">Cerrar</button>
                    <button onclick="openReservationFromPopup()" class="flex-1 bg-brand-dark text-white py-3 rounded-lg text-xs font-bold uppercase tracking-widest hover:bg-brand-copper transition-colors shadow-lg">Reservar Mesa</button>
                </div>
            </div>
        </div>
    </div>

    <!-- === POPUP ROOM (MEJORADO) === -->
    <div id="room-modal" class="fixed inset-0 z-[70] bg-black/60 backdrop-blur-md hidden opacity-0 transition-opacity duration-500 flex items-center justify-center p-4">
        <div class="bg-white w-full max-w-6xl h-[85vh] relative shadow-2xl overflow-hidden flex flex-col md:flex-row rounded-3xl animate-fade-in-up">
            <button onclick="closeRoomModal()" class="absolute top-6 right-6 z-50 bg-white/20 backdrop-blur text-white hover:bg-white hover:text-brand-dark w-12 h-12 rounded-full flex items-center justify-center transition-all text-xl shadow-lg border border-white/30">
                <i class="ph ph-x"></i>
            </button>
            
            <!-- Left: Big Image Slider -->
            <div class="w-full md:w-7/12 bg-gray-900 relative group h-64 md:h-full">
                <div id="room-slides-container" class="w-full h-full relative overflow-hidden">
                    <!-- Overlay gradient for text readability -->
                    <div class="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent z-10 pointer-events-none"></div>
                    <img src="https://images.unsplash.com/photo-1611892440504-42a792e24d32?q=80&w=1000" class="room-slide active transition-transform duration-[2000ms]" alt="Room View 1">
                    <img src="https://images.unsplash.com/photo-1582719478250-c89cae4dc85b?q=80&w=1000" class="room-slide transition-transform duration-[2000ms]" alt="Room View 2">
                    <img src="https://images.unsplash.com/photo-1631049307264-da0ec9d70304?q=80&w=1000" class="room-slide transition-transform duration-[2000ms]" alt="Room View 3">
                </div>
                <div class="absolute bottom-8 left-8 z-20 text-white">
                     <span class="bg-brand-copper/90 backdrop-blur px-3 py-1 text-[10px] font-bold uppercase tracking-widest rounded mb-3 inline-block" id="rm-subtitle">SUITE</span>
                     <h3 class="font-serif text-4xl md:text-5xl" id="rm-title">Nombre Habitación</h3>
                </div>
            </div>
            
            <!-- Right: Content & Booking -->
            <div class="w-full md:w-5/12 p-8 md:p-12 flex flex-col bg-white overflow-y-auto relative">
                <!-- Features Grid -->
                <div class="mb-8">
                    <p class="text-xs font-bold uppercase tracking-widest text-gray-400 mb-4">Amenidades Exclusivas</p>
                    <div class="grid grid-cols-2 gap-4">
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-wifi-high text-brand-copper text-lg"></i> <span>WiFi Fibra</span></div>
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-thermometer text-brand-copper text-lg"></i> <span>Climatización</span></div>
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-bed text-brand-copper text-lg"></i> <span>King Size Bed</span></div>
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-television text-brand-copper text-lg"></i> <span>Smart TV 55"</span></div>
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-coffee text-brand-copper text-lg"></i> <span>Coffee Station</span></div>
                        <div class="flex items-center gap-3 text-sm text-gray-600"><i class="ph ph-shower text-brand-copper text-lg"></i> <span>Rain Shower</span></div>
                    </div>
                </div>

                <div class="mb-8">
                     <p class="text-xs font-bold uppercase tracking-widest text-gray-400 mb-2">Descripción</p>
                     <p class="text-gray-600 font-light leading-relaxed text-sm" id="rm-desc">Descripción...</p>
                </div>
                
                <!-- Booking Form Box -->
                <div class="mt-auto bg-gray-50 p-8 rounded-2xl border border-gray-100 relative overflow-hidden">
                    <div class="absolute top-0 right-0 w-32 h-32 bg-brand-copper/5 rounded-full blur-3xl -mr-10 -mt-10"></div>
                    <h4 class="font-serif text-2xl text-brand-dark mb-6 relative z-10">Reservar Estadía</h4>
                    <div class="space-y-4 relative z-10">
                        <div>
                            <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Fecha de Llegada</label>
                            <input type="date" id="modal-date" class="w-full bg-white border border-gray-200 rounded-lg p-3 text-sm focus:border-brand-copper outline-none transition-colors">
                        </div>
                        <div>
                            <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Hora Estimada</label>
                            <input type="time" id="modal-time" class="w-full bg-white border border-gray-200 rounded-lg p-3 text-sm focus:border-brand-copper outline-none transition-colors">
                        </div>
                        <button onclick="submitRoomReservation()" class="btn-luxury w-full text-white py-4 font-bold uppercase tracking-[0.15em] text-xs rounded-lg shadow-md mt-2">
                            Confirmar Disponibilidad
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- === POPUP MENU === -->
    <div id="menu-modal" class="fixed inset-0 z-[70] bg-black/80 backdrop-blur-sm hidden opacity-0 transition-opacity duration-300 flex items-center justify-center p-4">
        <div class="relative bg-brand-cream w-full max-w-4xl h-[85vh] shadow-2xl rounded-3xl flex flex-col overflow-hidden">
            <button onclick="closeMenuModal()" class="absolute top-4 right-4 z-50 text-brand-dark text-3xl hover:text-brand-copper"><i class="ph ph-x"></i></button>
            <div class="flex-1 relative overflow-hidden bg-white">
                <div class="w-full h-full relative">
                    <div class="menu-page active p-12 text-center h-full flex flex-col justify-center">
                        <div class="border-2 border-brand-copper p-8 h-full rounded-2xl flex flex-col justify-center">
                            <h2 class="font-serif text-4xl text-brand-dark mb-2">Carta - Entradas</h2>
                            <div class="space-y-4 text-left max-w-lg mx-auto mt-8">
                                <div class="flex justify-between border-b border-gray-100 pb-2"><span class="font-serif text-lg">Ceviche de Ostiones</span><span class="text-brand-copper">$14.000</span></div>
                                <div class="flex justify-between border-b border-gray-100 pb-2"><span class="font-serif text-lg">Carpaccio de Res</span><span class="text-brand-copper">$12.500</span></div>
                            </div>
                        </div>
                    </div>
                    <div class="menu-page p-12 text-center h-full flex flex-col justify-center">
                        <div class="border-2 border-brand-copper p-8 h-full rounded-2xl flex flex-col justify-center">
                            <h2 class="font-serif text-4xl text-brand-dark mb-2">Carta - Fondo</h2>
                            <div class="space-y-4 text-left max-w-lg mx-auto mt-8">
                                <div class="flex justify-between border-b border-gray-100 pb-2"><span class="font-serif text-lg">Congrio a lo Pobre</span><span class="text-brand-copper">$18.900</span></div>
                                <div class="flex justify-between border-b border-gray-100 pb-2"><span class="font-serif text-lg">Asado de Tira</span><span class="text-brand-copper">$22.000</span></div>
                            </div>
                        </div>
                    </div>
                </div>
                <button onclick="prevMenuPage()" class="absolute left-4 top-1/2 -translate-y-1/2 text-brand-dark hover:text-brand-copper text-4xl"><i class="ph ph-caret-left"></i></button>
                <button onclick="nextMenuPage()" class="absolute right-4 top-1/2 -translate-y-1/2 text-brand-dark hover:text-brand-copper text-4xl"><i class="ph ph-caret-right"></i></button>
            </div>
        </div>
    </div>

    <!-- 4. POPUP COMPRA DE TICKETS (90% pantalla) -->
    <div id="ticket-modal" class="fixed inset-0 z-[100] bg-black/60 backdrop-blur-md hidden flex items-center justify-center transition-opacity duration-300">
        <!-- Contenedor 90% ancho y alto -->
        <div class="w-[90%] h-[90%] bg-white rounded-3xl relative overflow-hidden flex flex-col shadow-2xl animate-scale-in" onclick="event.stopPropagation()">
            
            <button onclick="closeTicketModal()" class="absolute top-6 right-6 z-50 bg-gray-100 hover:bg-gray-200 text-brand-dark w-12 h-12 rounded-full flex items-center justify-center transition-all text-xl shadow-sm">
                <i class="ph ph-x"></i>
            </button>

            <div class="bg-brand-dark text-white p-8 md:p-12 flex-shrink-0 relative overflow-hidden">
                <div class="absolute top-0 right-0 w-64 h-64 bg-brand-copper rounded-full mix-blend-multiply filter blur-[80px] opacity-20"></div>
                <div class="relative z-10">
                    <span class="text-brand-lightCopper font-bold tracking-[0.2em] text-xs uppercase mb-3 block">Compra de Entradas</span>
                    <h2 class="font-serif text-4xl md:text-5xl mb-2" id="tm-event-title">Nombre del Evento</h2>
                    <p class="text-white/60 text-lg font-light" id="tm-event-price-display">Valor Unitario: $0</p>
                </div>
            </div>

            <div class="flex-1 overflow-y-auto p-8 md:p-12 ticket-scroll bg-gray-50">
                <div class="max-w-5xl mx-auto">
                    <!-- Paso 1 -->
                    <div class="mb-12">
                        <label class="block text-brand-dark font-bold uppercase tracking-widest text-xs mb-4">Cantidad de Tickets</label>
                        <div class="flex items-center gap-6 bg-white p-4 rounded-2xl w-max shadow-sm border border-gray-100">
                            <button onclick="adjustTicketQuantity(-1)" class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center hover:bg-brand-dark hover:text-white transition-colors"><i class="ph ph-minus"></i></button>
                            <input type="number" id="tm-quantity" value="1" min="1" max="10" readonly class="w-16 text-center text-3xl font-serif font-bold bg-transparent focus:outline-none text-brand-dark">
                            <button onclick="adjustTicketQuantity(1)" class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center hover:bg-brand-dark hover:text-white transition-colors"><i class="ph ph-plus"></i></button>
                        </div>
                    </div>

                    <!-- Paso 2 -->
                    <div>
                        <h3 class="font-serif text-2xl text-brand-dark mb-8 flex items-center gap-4">
                            <span class="w-10 h-10 rounded-full bg-brand-copper text-white text-sm flex items-center justify-center font-sans font-bold shadow-lg">2</span>
                            Datos de los Asistentes
                        </h3>
                        <div id="tm-attendees-list" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            <!-- Inputs JS -->
                        </div>
                    </div>
                </div>
            </div>

            <div class="bg-white border-t border-gray-100 p-6 md:p-8 flex flex-col md:flex-row justify-between items-center gap-6 flex-shrink-0 shadow-[0_-5px_20px_rgba(0,0,0,0.02)]">
                <div class="text-center md:text-left">
                    <p class="text-[10px] text-gray-400 uppercase tracking-widest mb-1 font-bold">Total a Pagar</p>
                    <p class="text-4xl font-serif font-bold text-brand-dark" id="tm-total-price">$0</p>
                </div>
                
                <button onclick="processPayment()" class="bg-[#d9294e] hover:bg-[#b01e3c] text-white px-12 py-5 rounded-full font-bold uppercase tracking-widest flex items-center gap-4 transition-all shadow-xl hover:shadow-2xl hover:-translate-y-1 w-full md:w-auto justify-center group">
                    <span>Pagar con Webpay</span>
                    <i class="ph ph-credit-card text-xl group-hover:scale-110 transition-transform"></i>
                </button>
            </div>
        </div>
    </div>

    <!-- === POPUP COTIZADOR === -->
    <div id="quote-modal" class="fixed inset-0 z-[70] bg-black/80 backdrop-blur-sm hidden opacity-0 transition-opacity duration-300 flex items-center justify-center p-4">
        <div class="bg-white w-full max-w-5xl max-h-[90vh] overflow-y-auto rounded-3xl shadow-2xl relative flex flex-col md:flex-row">
            <button onclick="closeQuoteModal()" class="absolute top-4 right-4 z-50 text-gray-500 hover:text-brand-dark text-2xl"><i class="ph ph-x"></i></button>
            <div class="hidden md:block w-1/3 bg-brand-dark relative">
                <img id="quote-sidebar-img" src="https://images.unsplash.com/photo-1519167758481-83f550bb49b3?q=80&w=1000" class="absolute inset-0 w-full h-full object-cover opacity-60">
                <div class="relative z-10 p-10 text-white h-full flex flex-col justify-end bg-gradient-to-t from-black/90 to-transparent">
                    <h3 class="font-serif text-3xl mb-3" id="quote-sidebar-title">Evento Social</h3>
                    <p class="text-sm opacity-80 font-light leading-relaxed">Personalizamos cada detalle para crear experiencias inolvidables.</p>
                </div>
            </div>
            <div class="w-full md:w-2/3 p-10 md:p-14">
                <h2 class="font-serif text-3xl text-brand-dark mb-8">Cotizar Evento</h2>
                <form class="space-y-8" onsubmit="event.preventDefault(); showQuoteSuccess();">
                    <div>
                        <label class="block text-xs font-bold text-gray-400 uppercase tracking-widest mb-4">Seleccione Salón</label>
                        <div class="grid grid-cols-2 gap-4">
                            <label class="cursor-pointer border border-gray-200 p-5 rounded-xl hover:border-brand-copper transition-all has-[:checked]:border-brand-copper has-[:checked]:bg-brand-cream/50 has-[:checked]:shadow-md">
                                <input type="radio" name="salon" value="atacama" class="hidden" onchange="updateQuoteImages('atacama')">
                                <div class="font-serif text-lg text-brand-dark mb-1">Salón Atacama</div>
                                <div class="text-xs text-gray-400 font-medium">200 pax • Conferencias</div>
                            </label>
                            <label class="cursor-pointer border border-gray-200 p-5 rounded-xl hover:border-brand-copper transition-all has-[:checked]:border-brand-copper has-[:checked]:bg-brand-cream/50 has-[:checked]:shadow-md">
                                <input type="radio" name="salon" value="copiapo" class="hidden" onchange="updateQuoteImages('copiapo')">
                                <div class="font-serif text-lg text-brand-dark mb-1">Salón Copiapó</div>
                                <div class="text-xs text-gray-400 font-medium">80 pax • Reuniones</div>
                            </label>
                        </div>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                        <div class="relative">
                            <input type="date" required class="peer w-full border-b border-gray-300 py-2 placeholder-transparent focus:outline-none focus:border-brand-copper transition-colors bg-transparent">
                            <label class="absolute left-0 -top-3.5 text-xs font-bold text-brand-copper transition-all peer-placeholder-shown:text-base peer-placeholder-shown:text-gray-400 peer-placeholder-shown:top-2 peer-focus:-top-3.5 peer-focus:text-xs peer-focus:text-brand-copper uppercase tracking-widest">Fecha Estimada</label>
                        </div>
                        <div class="relative">
                            <input type="email" placeholder="email@empresa.com" required class="peer w-full border-b border-gray-300 py-2 placeholder-transparent focus:outline-none focus:border-brand-copper transition-colors bg-transparent">
                            <label class="absolute left-0 -top-3.5 text-xs font-bold text-brand-copper transition-all peer-placeholder-shown:text-base peer-placeholder-shown:text-gray-400 peer-placeholder-shown:top-2 peer-focus:-top-3.5 peer-focus:text-xs peer-focus:text-brand-copper uppercase tracking-widest">Email Contacto</label>
                        </div>
                    </div>
                    <button type="submit" class="btn-luxury w-full text-white py-4 font-bold uppercase tracking-widest rounded-xl shadow-lg mt-4">
                        Solicitar Cotización Formal
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- LIGHTBOX -->
    <div id="lightbox" class="fixed inset-0 z-[80] bg-black/95 hidden opacity-0 transition-opacity duration-300 flex items-center justify-center p-4" onclick="closeLightbox()">
        <img id="lightbox-img" src="" class="max-w-full max-h-[90vh] object-contain shadow-2xl rounded-lg">
        <button class="absolute top-6 right-6 text-white text-4xl hover:text-brand-copper">&times;</button>
    </div>

    <!-- CHATBOT -->
    <div id="chatbot-view" class="fixed inset-0 z-[60] bg-brand-cream transform translate-y-full transition-transform duration-500 ease-in-out flex flex-col">
        <div class="bg-brand-dark text-white p-6 flex justify-between items-center shadow-lg">
            <div class="flex items-center gap-4"><div class="w-10 h-10 rounded-full bg-brand-copper flex items-center justify-center"><i class="ph ph-robot text-2xl"></i></div><div><h3 class="font-serif text-xl font-bold">Asistente Atacama</h3><p class="text-xs text-brand-gray flex items-center gap-1"><span class="w-2 h-2 bg-green-500 rounded-full block"></span> En línea ahora</p></div></div>
            <button onclick="toggleChatbot()" class="text-white/70 hover:text-white"><i class="ph ph-x text-3xl"></i></button>
        </div>
        <div id="chat-messages" class="flex-1 overflow-y-auto p-6 space-y-4 bg-gray-50"></div>
        <div class="p-4 bg-white border-t border-gray-200">
            <form onsubmit="handleChatSubmit(event)" class="flex gap-2 relative"><input type="text" id="chat-input" placeholder="Consulta..." class="w-full bg-gray-100 text-brand-dark rounded-full px-6 py-4 focus:outline-none pr-12 font-light"><button type="submit" class="absolute right-2 top-2 bottom-2 w-10 h-10 bg-brand-copper text-white rounded-full flex items-center justify-center hover:bg-brand-dark"><i class="ph ph-paper-plane-right text-lg"></i></button></form>
        </div>
    </div>

    <!-- Scripts -->
    <script>
        // --- LOGICA MODAL COMPRA DE TICKETS ---
        let currentEventPrice = 0;
        let currentEventName = '';

        function openTicketModal(eventName, eventPrice) {
            currentEventName = eventName;
            currentEventPrice = eventPrice;
            document.getElementById('tm-event-title').innerText = eventName;
            document.getElementById('tm-event-price-display').innerText = 'Valor Unitario: ' + formatCurrency(eventPrice);
            document.getElementById('tm-quantity').value = 1;
            updateAttendeeInputs(1);
            const modal = document.getElementById('ticket-modal');
            modal.classList.remove('hidden');
            // Animar entrada
            const content = modal.querySelector('div');
            content.classList.remove('scale-95', 'opacity-0');
            content.classList.add('scale-100', 'opacity-100');
        }

        function closeTicketModal() {
            const modal = document.getElementById('ticket-modal');
            modal.classList.add('hidden');
        }

        function adjustTicketQuantity(change) {
            const input = document.getElementById('tm-quantity');
            let val = parseInt(input.value) + change;
            if (val < 1) val = 1;
            if (val > 10) val = 10;
            input.value = val;
            updateAttendeeInputs(val);
        }

        function updateAttendeeInputs(qty) {
            const container = document.getElementById('tm-attendees-list');
            container.innerHTML = '';
            for (let i = 1; i <= qty; i++) {
                const div = document.createElement('div');
                div.className = 'bg-white border border-gray-100 p-5 rounded-2xl shadow-sm hover:shadow-md transition-all hover:-translate-y-1';
                div.innerHTML = `
                    <div class="flex items-center justify-between mb-4 border-b border-gray-50 pb-2">
                        <span class="font-bold text-[10px] text-brand-copper uppercase tracking-widest">Ticket #${i}</span>
                        <i class="ph ph-ticket text-brand-dark text-lg opacity-50"></i>
                    </div>
                    <div class="space-y-4">
                        <div class="relative">
                            <input type="text" class="attendee-name peer w-full border-b border-gray-200 py-3 text-sm focus:outline-none focus:border-brand-copper bg-transparent placeholder-transparent" placeholder="Nombre">
                            <label class="absolute left-0 -top-2.5 text-[10px] font-bold text-gray-400 uppercase tracking-wide transition-all peer-placeholder-shown:text-sm peer-placeholder-shown:top-3 peer-focus:-top-2.5 peer-focus:text-[10px] peer-focus:text-brand-copper pointer-events-none">Nombre Completo</label>
                        </div>
                        <div class="relative">
                            <input type="text" class="attendee-rut peer w-full border-b border-gray-200 py-3 text-sm focus:outline-none focus:border-brand-copper bg-transparent placeholder-transparent" placeholder="RUT">
                            <label class="absolute left-0 -top-2.5 text-[10px] font-bold text-gray-400 uppercase tracking-wide transition-all peer-placeholder-shown:text-sm peer-placeholder-shown:top-3 peer-focus:-top-2.5 peer-focus:text-[10px] peer-focus:text-brand-copper pointer-events-none">RUT / Pasaporte</label>
                        </div>
                    </div>
                `;
                container.appendChild(div);
            }
            const total = qty * currentEventPrice;
            document.getElementById('tm-total-price').innerText = formatCurrency(total);
        }

        // --- PROCESAR PAGO CON TRANSBANK (REST) ---
        function processPayment() {
            const names = document.querySelectorAll('.attendee-name');
            const ruts = document.querySelectorAll('.attendee-rut');
            let isValid = true;
            names.forEach(input => { if(!input.value) isValid = false; });
            ruts.forEach(input => { if(!input.value) isValid = false; });

            if (!isValid) {
                Swal.fire({ title: 'Datos Incompletos', text: 'Por favor complete todos los datos de los asistentes.', icon: 'warning', confirmButtonColor: '#c17c5b' });
                return;
            }

            const total = parseInt(document.getElementById('tm-quantity').value) * currentEventPrice;
            
            const btn = document.querySelector("button[onclick='processPayment()']");
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="ph ph-spinner animate-spin text-xl"></i> Procesando...';
            btn.disabled = true;

            const formData = new FormData();
            formData.append('action', 'initiate_transbank_payment');
            formData.append('amount', total);
            formData.append('event_name', currentEventName);

            // Fetch to backend which communicates with Transbank API
            fetch('index_has_process.php', { method: 'POST', body: formData })
            .then(res => res.json())
            .then(resp => {
                btn.innerHTML = originalText;
                btn.disabled = false;
                
                if(resp.status === 'success') {
                    // Redirect to Webpay
                    // We create a form dynamically and submit it to the URL returned by Transbank
                    // Webpay REST requires submitting token_ws to the URL
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = resp.data.url;

                    const tokenInput = document.createElement('input');
                    tokenInput.type = 'hidden';
                    tokenInput.name = 'token_ws';
                    tokenInput.value = resp.data.token;

                    form.appendChild(tokenInput);
                    document.body.appendChild(form);
                    form.submit();
                } else {
                    Swal.fire('Error', resp.message, 'error');
                }
            })
            .catch(err => {
                btn.innerHTML = originalText;
                btn.disabled = false;
                console.error(err);
                Swal.fire('Error de Conexión', 'No se pudo contactar con el servidor de pagos.', 'error');
            });
        }

        function formatCurrency(amount) {
            return new Intl.NumberFormat('es-CL', { style: 'currency', currency: 'CLP' }).format(amount);
        }

        // --- FUNCIONES GENÉRICAS ---
        function showReservationAlert() {
            Swal.fire({
                title: 'Verificando Disponibilidad',
                text: 'Conectando con el sistema de reservas...',
                timer: 2000,
                timerProgressBar: true,
                didOpen: () => { Swal.showLoading() }
            }).then(() => {
                Swal.fire({ title: 'Disponibilidad Confirmada', text: 'Redirigiendo al formulario de pago seguro.', icon: 'success', confirmButtonColor: '#c17c5b', confirmButtonText: 'Continuar' });
            });
        }

        function scrollSuites(direction) {
            const container = document.getElementById('suites-container');
            const scrollAmount = container.clientWidth * 0.8;
            container.scrollBy({ left: direction === 'left' ? -scrollAmount : scrollAmount, behavior: 'smooth' });
        }

        // Preloader Logic
        window.addEventListener('load', () => {
            document.getElementById('preloader').classList.add('loaded');
        });

        // Slider Auto Init
        function initSlider(id) {
            let current = 0;
            const images = document.querySelectorAll(`#${id} .slider-image, #${id} .salon-slide`);
            if(images.length === 0) return;
            setInterval(() => {
                images[current].classList.remove('active');
                current = (current + 1) % images.length;
                images[current].classList.add('active');
            }, 4000);
        }
        initSlider('gastronomy-slider');
        initSlider('salon-slider-container');
        
        // ... (resto de funciones como openQuoteModal, etc. se asumen presentes o se añaden si faltan)
        function openQuoteModal() { document.getElementById('quote-modal').classList.remove('hidden'); setTimeout(()=>document.getElementById('quote-modal').classList.remove('opacity-0'),10); }
        function closeQuoteModal() { document.getElementById('quote-modal').classList.add('opacity-0'); setTimeout(()=>document.getElementById('quote-modal').classList.add('hidden'),300); }
        function showQuoteSuccess() { closeQuoteModal(); Swal.fire({title:'Enviado', icon:'success', confirmButtonColor:'#c17c5b'}); }
        function toggleChatbot() { document.getElementById('chatbot-view').classList.toggle('translate-y-full'); }
        function openMenuModal() { document.getElementById('menu-modal').classList.remove('hidden'); setTimeout(()=>document.getElementById('menu-modal').classList.remove('opacity-0'),10); }
        function closeMenuModal() { document.getElementById('menu-modal').classList.add('opacity-0'); setTimeout(()=>document.getElementById('menu-modal').classList.add('hidden'),300); }
        
        let currentMenuPage = 0; 
        const menuPages = document.getElementsByClassName('menu-page');
        function nextMenuPage() { currentMenuPage=(currentMenuPage+1)%menuPages.length; updateMenuPages(); }
        function prevMenuPage() { currentMenuPage=(currentMenuPage-1+menuPages.length)%menuPages.length; updateMenuPages(); }
        function updateMenuPages() { for(let i=0; i<menuPages.length; i++) menuPages[i].classList.remove('active'); menuPages[currentMenuPage].classList.add('active'); }
        function openLightbox(src) { const lb = document.getElementById('lightbox'); const img = document.getElementById('lightbox-img'); img.src = src; lb.classList.remove('hidden'); setTimeout(() => lb.classList.remove('opacity-0'), 10); }
        function closeLightbox() { const lb = document.getElementById('lightbox'); lb.classList.add('opacity-0'); setTimeout(() => lb.classList.add('hidden'), 300); }

        // Room Modal Functions
        let currentRoomIndex = 0;
        let roomSliderInterval;
        const roomImages = document.getElementsByClassName('room-slide');

        function openRoomModal(title, subtitle, desc, autoScroll = false) {
            const modal = document.getElementById('room-modal');
            document.getElementById('rm-title').innerText = title;
            document.getElementById('rm-subtitle').innerText = subtitle;
            document.getElementById('rm-desc').innerText = desc;
            currentRoomIndex = 0; showRoomSlide(0); startRoomSlider();
            modal.classList.remove('hidden');
            setTimeout(() => modal.classList.remove('opacity-0'), 10);
            if(autoScroll) { setTimeout(() => { const inputs = modal.querySelector('input'); if(inputs) inputs.scrollIntoView({behavior: 'smooth', block: 'center'}); }, 300); }
        }
        function closeRoomModal() { const modal = document.getElementById('room-modal'); stopRoomSlider(); modal.classList.add('opacity-0'); setTimeout(() => modal.classList.add('hidden'), 500); }
        function startRoomSlider() { stopRoomSlider(); roomSliderInterval = setInterval(nextRoomSlide, 5000); }
        function stopRoomSlider() { clearInterval(roomSliderInterval); }
        function showRoomSlide(n) { if (roomImages.length === 0) return; if (n >= roomImages.length) currentRoomIndex = 0; if (n < 0) currentRoomIndex = roomImages.length - 1; for (let i = 0; i < roomImages.length; i++) { roomImages[i].classList.remove('active'); } roomImages[currentRoomIndex].classList.add('active'); }
        function nextRoomSlide() { currentRoomIndex++; showRoomSlide(currentRoomIndex); }
        function submitRoomReservation() { closeRoomModal(); Swal.fire({ title: 'Solicitud Recibida', icon: 'success', confirmButtonColor: '#c17c5b' }); }

    </script>
</body>
</html>